#!/usr/bin/perl
################################################################
#																					#
#  package ACTINIC_PXML - ACTINIC specific parser					#
#																					#
#  Separated from ACTINIC package in v6								#
#																					#
#  This module contains the ACTINIC_PXML object description.	#
#  This object is the Actinic specific event driven extension	#
#	of the core PXML object.												#
#	The basic member functions are inherited form the parent 	#
# 	PXML object. The below described functions are the event		#
#	(XML tag) handler functions.											#
#																					#
#	Each event hadler is called twice for each XML tag (open		#
#	and end tag). These calls can be identified by the tag name.#
#	Each event handler receives the following arguments:			#
#  	$sTag 			- tag name											#
#  	$sInsideText 	- reference to text between start and end #
#     $ParameterHash - hash of parameters,							#
#     $sId 				- current tag prefix,							#
#     $sFullTag 		- full text of current tag;					#
#																					#
#	Author: Zoltan Magyar													#
#																					#
#  Copyright (c)2002 Actinic Software Ltd 							#
#																					#
################################################################

package ACTINIC_PXML;
push (@INC, "cgi-bin");
<Actinic:Variable Name="IncludePathAdjustment"/>

require <Actinic:Variable Name="PXMLPackage"/>;
#
# Version
#
$ACTINIC_PXML::prog_name = 'ACTINIC_PXML.pm';	# Program Name
$ACTINIC_PXML::prog_name = $ACTINIC_PXML::prog_name;	# remove compiler warning
$ACTINIC_PXML::prog_ver = '$Revision: 29820 $ ';		# program version
$ACTINIC_PXML::prog_ver = substr($ACTINIC_PXML::prog_ver, 11); # strip the revision information
$ACTINIC_PXML::prog_ver =~ s/ \$//;					# and the trailers

#use constant Version => "1.0, (PXML: " . PXML->Version . ")";

############################################################
#
#  ACTINIC_PXML->new() - constructor for ACTINIC_PXML class
#
#  A very standard constructor. Derives itself from PXML and
#	set up the event handler functions.
#	See object header and PXML.pm for more details.
#
#  Author:	Ryszard Zybert  Dec  1 18:15:36 GMT 1999
#
#  Copyright (c) Actinic Software Ltd 1999
#
############################################################

use vars qw(@ISA);
@ISA = qw(PXML);											# inheritance

sub new
	{
	my $Proto = shift;
	my $Class = ref($Proto) || $Proto;
	my $self  = $Class->SUPER::new();				# dont pass arguments, we can use Set()

	bless ($self, $Class);
	#
	# Set up event handler functions
	#
	$self->Set(
				  ID						=>	'Actinic:',													# default prefix
				  XMLERROR				=> "<br>" . ACTINIC::GetPhrase(-1, 1972, $::g_sRequiredColor) . "<b>". ACTINIC::GetPhrase(-1, 218) . "</b>" . ACTINIC::GetPhrase(-1, 1970) . "<br>",
				  MAINFRAME				=>	sub { $self->MainFrameTagHandler(@_)		},		# handle url of main frame
				  PRICES					=> sub { $self->PriceTagHandler(@_)				},		# price tag
				  PRICE_EXPLANATION	=> sub { $self->ExplanationTagHandler(@_)		},		# price explanation tag				  
				  VAR						=> sub { $self->VarTagHandler(@_)				},		# var tag
				  SECTION				=> sub { $self->SectionTagHandler(@_)			},		# section tag
				  ADDRESSES				=> sub { $self->AddressTagHandler(@_)			},		# addresses tag
				  UNREG					=> sub { $self->UnregTagHandler(@_)				},		# unregistered user tag
				  IGNORE					=> sub { $self->IgnoreTagHandler(@_)			},		# IGNORE tag (deletes text)
				  REMOVE					=> sub { $self->RemoveTagHandler(@_)			},		# Remove tag (deletes text if its parameter not defined)
				  NOTINB2B				=> sub { $self->NotInB2BTagHandler(@_)			},		# NOTINB2B tag (deletes text)
				  ONLYINB2B				=> sub { $self->OnlyInB2BTagHandler(@_)			},		# NOTINB2B tag (deletes text)
				  BASEHREF				=> sub { $self->BaseHrefTagHandler(@_)			},		# BASEHREF tag
				  DEFAULT				=> sub { $self->DefaultTagHandler(@_)			},		# unknown tags here
				  XMLTEMPLATE			=> sub { $self->XMLTemplateTagHandler(@_)		},		# XML template tag handler
				  CARTERROR				=> sub { $self->CartErrorTagHandler(@_)		}, 	# Cart error placeholder
				  RETAIL_ONLY_SEARCH => sub { $self->RetailOnlySearchTagHandler(@_)},
				  LOCATION						=> sub { $self->LocationTagHandler(@_)					},	# LOCATION tags
				  EXTRAFOOTERTEXT				=> sub { $self->ExtraFooterTagHandler(@_)				},	# EXTRAFOOTERTEXT tags
				  EXTRACARTTEXT				=> sub { $self->ExtraCartTagHandler(@_)				},	# EXTRACARTTEXT tags
				  EXTRACARTBASEPLUSPERTEXT	=> sub { $self->ExtraCartBasePlusPerTagHandler(@_)	},	# EXTRACARTBASEPLUSPERTEXT tags
				  EXTRASHIPPINGTEXT			=> sub { $self->ExtraShippingTagHandler(@_)			},	# EXTRASHIPPINGTEXT tags
				  BASEPLUSPERRATEWARNING	=> sub { $self->BasePlusPerInfoTagHandler(@_)		},	# BASEPLUSPERRATEWARNING tags
				  DEFAULTTAXZONEMESSAGE		=> sub { $self->DefaultTaxZoneMessageTagHandler(@_)},	# DEFAULTTAXZONEMESSAGE tags
				  SHOWFORPRICESCHEDULE		=> sub { $self->ShowForPriceScheduleTagHandler(@_)	},		# ShowForPriceSchedule tags
				  COOKIECHECK					=> sub { $self->AddCookieCheck(@_)						}, # Cookie checking code placeholder
				  INVOICEADDRESSDETAILS		=> sub { $self->AddressDetailTagHandler(@_)			}, # invoice address details
				  DELIVERADDRESSDETAILS		=> sub { $self->AddressDetailTagHandler(@_)			}, # delivery address details
				  BOTHADDRESSDETAILS			=> sub { $self->AddressDetailTagHandler(@_)			}, # both address details
				  CHECKOUTURL					=> sub { $self->CheckoutURLTagHandler(@_)				}, # checkout URL
				  SHIPPINGDETAILS				=> sub { $self->ShippingDetailsTagHandler(@_)		}, # shipping details
				  TAXDETAILS					=> sub { $self->TaxDetailsTagHandler(@_)				}, # tax details
				  COMFIRMTAXEXEMPTION		=> sub { $self->ConfirmTaxExemptionTagHandler(@_)	}, # tax details
				  TAXUSERDEFINED				=> sub { $self->TaxUserDefinedTagHandler(@_)			}, # tax details
				  TAXCHANGECELL				=> sub { $self->TaxChangeCellTagHandler(@_)			}, # tax details
				  COUPON							=> sub { $self->CouponTagHandler(@_)					}, # coupon
				  TANDC							=> sub { $self->TermsTagHandler(@_)						}, # terms and conditions
				  PAYMENT						=> sub { $self->PaymentTagHandler(@_)					}, # payment
				  PAYMENTCARDFIELDS			=> sub { $self->PaymentCardFieldsTagHandler(@_)		}, # payment card fields
				  PAYMENTCARDTYPE				=> sub { $self->PaymentCardTypeTagHandler(@_)		}, # payment card type
				  DIGITALDOWNLOADINFO		=> sub { $self->DigitalDownloadInfoTagHandler(@_)	}, # digital download info
				  PAYMENTNOPAYMENT			=> sub { $self->PaymentNoPaymentTagHandler(@_)		}, # payment
				  ACCOUNT						=> sub { $self->AccountTagHandler(@_)					}, # account tag handler
				  GENERAL						=> sub { $self->GeneralTagHandler(@_)					}, # general tag handler
				  DELIVERPOSTALCODE			=> sub { $self->DeliverPostalCodeTagHandler(@_)		}, # postal code tag handler
				  DELIVERRESIDENTIAL			=> sub { $self->DeliverResidentialTagHandler(@_)	}, # residential tag handler
				  SEARCHFIELD					=> sub { $self->SearchFieldTagHandler(@_)				},	# search field tag handler
				  SEARCHFIELD_END				=> sub { $self->SearchFieldEndTagHandler(@_)			},	# end of search field tag handler
				  STATICSEARCHFIELD			=> sub { $self->IgnoreTagHandler(@_)					},	# delete static search fields using ignore handler
				 );
	$self->Set(@_);
	return $self;
	}

############################################################
#
#  ExplanationTagHandler - process the price explanation tag
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Zoltan Magyar  Feb  22 23:22 GMT 2001
#
#  Copyright (c) Actinic Software Ltd (2001)
#
############################################################

sub ExplanationTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag =~ /^\// )									# If End tag
		{
		return "";
		}

	if( $Self->{CurrentSectionBlob} )				# If section blob set				
		{
		my @Response;
		
			@Response = ACTINIC::GetProduct($ParameterHash->{PROD_REF}, $Self->{CurrentSectionBlob},
								ACTINIC::GetPath());			# get this product object		
		my ($Status, $Message, $pProduct) = @Response;
		if ($Status != $::SUCCESS)
			{
			return "";
			}													# If any problem, forget it

		if (defined $$pProduct{PRICES})
			{
			#
			# Need to work out which prices to show
			#
			my ($bShowRetailPrices, $bShowCustomerPrices, $nAccountSchedule) = ACTINIC::DeterminePricesToShow();
			my $sComments;
			if ($nAccountSchedule == -1)
				{
				$nAccountSchedule = $ActinicOrder::RETAILID;
				}
			#
			# Look up the apropriate product or component comment
			#
			if (defined $ParameterHash->{COMPONENTID} &&
				 $ParameterHash->{COMPONENTID} != -1)
				{
				my $nComponentID = $ParameterHash->{COMPONENTID};
				#
				# Check that associated product pricing is used or not
				# and select the appropriate message
				#
				if ($pProduct->{COMPONENTS}[$nComponentID][$::CBIDX_ASSOCPRODPRICE] == 1)	# if associated product prices are used
					{
					#
					# Get the message from associated product hash
					#
					my $Assoc = $pProduct->{COMPONENTS}[$nComponentID][$::CBIDX_PERMUTATIONS][0][$::PBIDX_ASSOCIATEDPROD];
					#
					# Check if the associated product blob is defined
					# It is not defined when the associated product is out of stock
					# or the permutation is invalid. The [3][0][1] item contains
					# '+' or '-' strings in these cases (IOW no explanation hash).
					#
					if (ref $Assoc eq 'HASH')
						{
						$sComments = $$Assoc{PRICE_COMMENTS}->{$nAccountSchedule};
						}
					}
				elsif (defined $pProduct->{COMPONENTS}[$nComponentID][$::CBIDX_EXPLANATION] &&
					 ref($pProduct->{COMPONENTS}[$nComponentID][$::CBIDX_EXPLANATION]) eq 'HASH')
					{
					$sComments = $pProduct->{COMPONENTS}[$nComponentID][$::CBIDX_EXPLANATION]->{$nAccountSchedule};
					}
				}
			else
				{
				$sComments = $pProduct->{'PRICE_COMMENTS'}->{$nAccountSchedule};
				}
			if ($sComments ne '')
				{
				$$sInsideText = ACTINIC::GetPhrase(-1, 2296). $sComments . ACTINIC::GetPhrase(-1, 2297);	# wrap non empty text with BLOCKQUOTES
				}
			}
		}
	return "";
	}

############################################################
#
#  RetailOnlySearchTagHandler - process the search tag
#
#  Note: <Actinic:Actinic:RETAIL_ONLY_SEARCH/> acts globally
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
############################################################

sub RetailOnlySearchTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	my $sDigest = $ACTINIC::B2B->Get('UserDigest');	# get the user identifying digest
	if ($sTag !~ /^\//)									# If not End tag
		{
		if ($sDigest)										# If there is a user remove tags and text
			{
			if (ref($sInsideText))
				{
				$$sInsideText = "";
				}
			}
		}
	else														# here on the second call (the end tag)
		{
		return ('');										# on the second pass, return nothing to prevent duplicate entries
		}
	#
	# If the buyer exists and does not see retail prices, note that the search is retail prices only
	#
	my $sRetailMessage = ACTINIC::GetPhrase(-1, 357);

	if ($sDigest)											# see which price schedule they use
		{
		my ($Status, $sMessage, $pBuyer) = ACTINIC::GetBuyer($sDigest, ACTINIC::GetPath()); # look up the buyer
		if ($Status == $::SUCCESS)
			{
			my $pAccount;
			($Status, $sMessage, $pAccount) = ACTINIC::GetCustomerAccount($pBuyer->{AccountID}, ACTINIC::GetPath()); # find the account information
			if ($Status == $::SUCCESS)
				{
				if ($pAccount->{PriceSchedule} == $ActinicOrder::RETAILID) # if we can confirm that this buyer is using the retail schedule
					{
					$sRetailMessage = '';				# the message is unnecessary
					}
				}
			}
		}

	return ($sRetailMessage);
	}

############################################################
#
#  SetAddressTabIndexes - Substitute NETQUOTEVAR:TABINDEX tags
#		with appropriate numerical value
#
#  Input:	$rsText			- reference to text to substitute
#           @arrPrefixes	- populated by rest of params with list of identifiers used for ordering
#
#  Returns: substituted string
#
#  Author:	Mike Purnell
#
############################################################

sub SetAddressTabIndexes
	{
	my ($rsText, @arrPrefixes) = @_;
	#
	# Tab ordering is done is prefix to identifier and then source order
	#
	my ($sPrefix, $nTabIndex, %hashTabIndexes);
	$nTabIndex = 1;
	foreach $sPrefix (@arrPrefixes)													# for each prefix (INVOICE or DELIVER)
		{
		my @arrMatches = $$rsText =~ /NETQUOTEVAR:TABINDEX$sPrefix\w+/gs;	# get array of matches
		my $sTabIndex;
		foreach $sTabIndex (@arrMatches)												# for each match, increment the tab order
			{
			$hashTabIndexes{$sTabIndex} = $nTabIndex++;
			}
		}
	#
	# Substitute the values
	#
	my ($nStatus, $sError, $sResult) = ACTINIC::TemplateString($$rsText, \%hashTabIndexes);
	return $sResult;
	}
	
#######################################################
#
# AccountTagHandler - Handle customer account addresses
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub AccountTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		my $sDigest = $ACTINIC::B2B->Get('UserDigest');
		#
		# Set the tab order if this contains all address fields
		#
		if ($ParameterHash->{'TYPE'} eq 'ROW') 
			{
			if (!$sDigest)
				{
				$$rsInsideText = '';
				return '';
				}
			my ($Status, $sMessage, $pBuyer, $pAccount) = ACTINIC::GetBuyerAndAccount($sDigest, ACTINIC::GetPath());
			if ($Status != $::SUCCESS)
				{
				if ($Status != $::NOTFOUND)
					{
					ACTINIC::ReportError($sMessage, ACTINIC::GetPath());	# if the error is not the missing digest the error should be reported
					}
				return("");
				}
			#
			# Get the lists of valid addresses
			#
			my ($pAddress, $plistValidAddresses, $plistValidInvoiceAddresses, $plistValidDeliveryAddresses);
			($Status, $sMessage, $plistValidInvoiceAddresses, $plistValidDeliveryAddresses) =
				ACTINIC::GetCustomerAddressLists($pBuyer, $pAccount, $::TRUE);
			if ($Status != $::SUCCESS)
				{
				return("");
				}
			my $sInvoiceHTML = $Self->GetAddressSelectionHTML($pAccount, $pBuyer, 'Invoice', $plistValidInvoiceAddresses);
			$$rsInsideText =~ s/<Actinic:ACCOUNTINVOICE \/>/$sInvoiceHTML/s;
			my $sDeliveryHTML = $Self->GetAddressSelectionHTML($pAccount, $pBuyer, 'Delivery', $plistValidDeliveryAddresses);
			$$rsInsideText =~ s/<Actinic:ACCOUNTDELIVER \/>/$sDeliveryHTML/s;
			}
		}
	return '';
	}

#######################################################
#
# GetAddressSelectionHTML - Get the html for customer account selection
#
# Input:	$pAccount				- ref to account object
#			$pBuyer					- ref to buyer object
#			$sAddressType			- address type 'Invoice' or 'Delivery'
#			$plistValidAddresses	- ref to list of valid addresses
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub GetAddressSelectionHTML
	{
	my ($Self, $pAccount, $pBuyer, $sAddressType, $plistValidAddresses) = @_;
	
	my $sNamePrefix = ($sAddressType eq 'Invoice') ? 'INVOICE' : 'DELIVER';
	my ($nRule, $sSelect, $Status, $sMessage, $pAddress);
	
	if ($pAccount->{$sAddressType . 'AddressRule'} == 1)	# The Customer rule overrides buyer rule
		{
		$nRule = 0;
		$sSelect = $pAccount->{$sAddressType . 'Address'};# Default (or fixed) address
		($Status, $sMessage, $pAddress) = ACTINIC::GetCustomerAddress($$pBuyer{AccountID}, $sSelect, ACTINIC::GetPath());
		if ($Status != $::SUCCESS)
			{
			ACTINIC::CloseCustomerAddressIndex(); # The customer index is left open for multiple access, so clean it up here
			return("");
			}
		}
	else
		{
		$nRule   = $pBuyer->{$sAddressType . 'AddressRule'};	# Address rule for this user
		$sSelect = $pBuyer->{$sAddressType . 'AddressID'};   # Default (or fixed) address
		if($nRule == 0 || ($nRule == 1 && $#$plistValidAddresses == 0))
			{
			$nRule = 0;
			$pAddress = $plistValidAddresses->[0];
			$sSelect  = $pAddress->{ID};
			}
		else
			{
			$sSelect = GetSelectedBuyerAddress($sAddressType, $plistValidAddresses, $sSelect);
			}
		}
	if ($sSelect == -1)										# if we don't have a default address defined
		{
		$sSelect = $plistValidAddresses->[0]->{ID};	# use first valid address
		}
	$Self->{$sAddressType . 'Rule'} = $nRule;
	my $sTableFormat   	= $Self->{Variables}->{ADDRESS_TABLE};
	my $sTitle         	= $Self->{Variables}->{'ADDRESS_TITLE' . $nRule};
	my $sTitle_1        	= $Self->{Variables}->{'ADDRESS_TITLE1' . $nRule};
	my $sForm				= '<TD>' . $Self->{Variables}->{'ADDRESS_FORM' . $nRule} . '</TD>';

	my $sAddressText = sprintf('<input type="hidden" id="id%sRule" value="%d">',
		$sAddressType, $nRule);

	if( $nRule == 0 )										# Rule 0 - fixed address
		{
		$sAddressText .= '<table><TR><TD style="vertical-align: top;">';					# Just a single cell
		$sAddressText .= sprintf($sForm,
										$sNamePrefix,
										 $sSelect,						# Address ID
										 $pAddress->{Name},			# Address text follows
										 $pAddress->{Line1},
										 $pAddress->{Line2},
										 $pAddress->{Line3},
										 $pAddress->{Line4},
										 $pAddress->{PostCode},
										 ACTINIC::GetCountryName($pAddress->{CountryCode}));
		$sAddressText .= '</TD></TR></table>';
		}
	else										# Rule 0 - fixed address
		{
		$sAddressText .= '<table>';					# Just a single cell
		my $nCount = 0;
		my $nRowCount = 0;
		my $sCh;
		foreach $pAddress (@$plistValidAddresses)
			{
			$sAddressText .= '<tr>';
			if( $pAddress->{ID} eq $sSelect )	# For Rule 1 check default address
				{
				$sCh = ' CHECKED';
				}
			else
				{
				$sCh = '';
				}

			$sAddressText .= sprintf($sForm,
											 ACTINIC::GetPhrase(-1, 301),
											 $sNamePrefix,
											 $pAddress->{ID},				# Address ID (for RADIO button)
											 $sCh,							# Optional 'CHECKED'
											 $pAddress->{Name},			# Address text follows
											 $pAddress->{Line1},
											 $pAddress->{Line2},
											 $pAddress->{Line3},
											 $pAddress->{Line4},
											 $pAddress->{PostCode},
											 ACTINIC::GetCountryName($pAddress->{CountryCode}));

			$sAddressText .= '</tr>';
			}
		$sAddressText .= '</table>';
		if ($nRule == 2)
			{
			my $sText = ($sNamePrefix eq 'INVOICE') ?
				ACTINIC::GetPhrase(-1, 303,ACTINIC::GetPhrase(-1, 304)) :
				ACTINIC::GetPhrase(-1, 303,ACTINIC::GetPhrase(-1, 305));

			$sAddressText .= sprintf($sTitle_1, $sText, $sNamePrefix, $sNamePrefix, ($sSelect) ? '' : 'checked');
			}
		}
	$sAddressText =~ s/<br>[,\s]*/<br>/gi;			# Remove leading commas
	$sAddressText =~ s/[,\s]*<br>/<br>/gi;			# Remove trailing commas
	$sAddressText =~ s/(<br>)+/<br>/gi;				# Remove blank lines
	return $sAddressText;
	}

############################################################
#
# GetSelectedBuyerAddress	- Get the selected buyer address
#
# Input: $sAddressType - type of address 'Invoice' or 'Delivery'
#        $plistValidAddresses - reference to list of valid addresses
#
# Returns: ID of address to select or empty string
#
############################################################

sub GetSelectedBuyerAddress
	{
	my ($sAddressType, $plistValidAddresses, $sDefaultID) = @_;
	
	my ($rhashContact, $sLocationKeyPrefix);
	if ($sAddressType eq 'Invoice')
		{
		$sLocationKeyPrefix = 'INVOICE';
		$rhashContact = \%::g_BillContact;
		}
	else
		{
		$sLocationKeyPrefix = 'DELIVERY';
		$rhashContact = \%::g_ShipContact;
		}
	#
	# If an address has already been selected, return it
	#
	if (exists $rhashContact->{ADDRESSSELECT})
		{
		return $rhashContact->{ADDRESSSELECT};
		}
	#
	# Use the location info to get the first matching address
	#
	my $sCountryCode = $::g_LocationInfo{$sLocationKeyPrefix . '_COUNTRY_CODE'};
	if ($sCountryCode eq '')							# if we don't know country 
		{
		return $sDefaultID;								# we can't match the address
		}
	my $sStateCode = $::g_LocationInfo{$sLocationKeyPrefix . '_REGION_CODE'};
	my $sFirstMatchingID = '';
	my $pAddress;
	foreach $pAddress (@$plistValidAddresses)
		{
		if ($pAddress->{CountryCode} eq $sCountryCode)
			{
			if ($sStateCode eq $ActinicOrder::UNDEFINED_REGION ||
				$pAddress->{StateCode} eq $sStateCode)
				{
				if ($pAddress->{ID} eq $sDefaultID)
					{
					return $pAddress->{ID};
					}
				elsif ($sFirstMatchingID eq '')
					{
					$sFirstMatchingID = $pAddress->{ID};
					}
				}
			}
		}
	if ($sFirstMatchingID ne '')
		{
		return $sFirstMatchingID;
		}
	return $sDefaultID;
	}

############################################################
#
#  AddressTagHandler - callback for addresses
#  Replaces <Actinic:ADDRESSES/> tag by address table
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Ryszard Zybert  Jan  3 16:44:37 GMT 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
#
############################################################

sub AddressTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	my $sDigest = $ACTINIC::B2B->Get('UserDigest');

	if( $sTag =~ /^\// )									# Ignore end-tags
		{
		return "";
		}
	if ($sDigest ne '' &&								# if this is a customer account
		$Self->{'InvoiceRule'} != 2 &&				# and they can't enter an invoice address
		$Self->{'DeliveryRule'} != 2)					# and they can't enter a delivery address
		{
		$$sInsideText = "";								# hide entry fields
		return '';
		}
	#
	# Set the tab order if this contains all address fields
	#
	$$sInsideText = SetAddressTabIndexes($sInsideText, 'INVOICE', 'DELIVER');
	return '';
	}


############################################################
#
#  sub VarTagHandler - callback for variables
#  Sets variables
#  There should be NAME and VALUE parameters
#  $Self->{Variables}->{name} is set to value
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Ryszard Zybert  Dec  7 20:58:25 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
#
############################################################

sub VarTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag !~ /^\// )
		{
		$Self->{Variables}->{$ParameterHash->{NAME}} = $ParameterHash->{VALUE};
		}
	return "";
	}

############################################################
#
#  sub CartErrorTagHandler - cart error tag handler
#
#  If any error message is set for the actual product reference
#  (in the ProdRef parameter) then replace the tag to the
#  error message.
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Zoltan Magyar, 5/14/2003
#
#  Copyright (c) Actinic Software Ltd 2003
#
############################################################

sub CartErrorTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag =~ /^\// )
		{
		return "";
		}
	#
	# Check if we have error message stored for the actual product reference
	#
	my $sErrorValue = $ACTINIC::B2B->GetXML("CartError_" . $ParameterHash->{ProdRef});
	if (defined $sErrorValue)
		{
		return $sErrorValue;
		}

	return "";
	}

############################################################
#
#  sub DefaultTagHandler - callback for unknown tags
#  Looks up the tag in the B2B XML Tags hash
#  If found - replaces the tag by it
#  (meaning the whole <Actinic:$sTag....> sequence)
#  If not doesn't -leaves everything untouched
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Ryszard Zybert  Dec  7 20:58:25 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
#
############################################################

sub DefaultTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	my $sXMLTag = $ACTINIC::B2B->GetXML($sTag);
	if (defined $sXMLTag)
		{
		return $sXMLTag;
		}
	return $sFullTag;
	}

############################################################
#
#  XMLTemplateTagHandler - callback for XMLTEMPLATE tags
#  Looks up the NAME parameter in the B2B XML Tags hash
#  If found - replaces the tag by it
#  (meaning the whole <Actinic:$sTag....>  </...> sequence)
#  If not doesn't - removes everything
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Ryszard Zybert  Dec  7 20:58:25 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
#
############################################################

sub XMLTemplateTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag =~ /^\// )									# Ignore end-tag completely
		{
		return "";
		}

	my $sTagname = $ParameterHash->{NAME};
	my $sXMLTag = $ACTINIC::B2B->GetXML($sTagname);

	if (defined $sXMLTag)
		{
		$$sInsideText = "";								# replace (i.e. don't leave enclosed text)
		return $sXMLTag;									# return replacement
		}
	$$sInsideText = "";
	return "";
	}

############################################################
#
#  sub RetailPriceTextTagHandler - callback for retail text
#  Sets XML variable
#  B2B->{XML}->{tag} is set to value of text between tags
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : empty string
#
############################################################


sub RetailPriceTextTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag !~ /^\// )									# Ignore end-tag completely
		{
		if(ref($sInsideText))							# If there is text store it in XML variable
			{
			$ACTINIC::B2B->SetXML($sTag, $$sInsideText);
			my $sDigest = $ACTINIC::B2B->Get('UserDigest');
			if ($sDigest)
				{
				$$sInsideText = "";						# Text not needed anymore for logged in
				}
			}
		}
	return "";												# Both tags also removed
	}

############################################################
#
#  DefaultRemovingTagHandler - callback for unknown tags
#  As DefaultTagHandler except that unknown tags are removed
#  together with all text between tags
#
#  Looks up the tag in the B2B XML Tags hash
#  If found - replaces the tag by it
#  (meaning the whole <Actinic:$sTag....> sequence)
#  If not removes everything
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Ryszard Zybert  Dec  7 20:58:25 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
#
############################################################

sub DefaultRemovingTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	my $sXMLTag = $ACTINIC::B2B->GetXML($sTag);
	if( defined($sXMLTag) )								# Don't touch text, return replacement tag
		{
		return $sXMLTag;
		}
	else														# Clear both text and tag
		{
		if( ref($sInsideText) )
			{
			$$sInsideText = "";
			}
		return "";
		}
	}

############################################################
#
#  IgnoreTagHandler
#  Remove text within the tag and the tag
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Ryszard Zybert  Jul 25 22:30:51 BST 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
#
############################################################

sub IgnoreTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	if( ref($sInsideText) )
		{
		$$sInsideText = "";
		}
	return "";
	}

############################################################
#
#  RemoveTagHandler
#  Remove text within the tag and the tag if the tag is not defined
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Zoltan Magyar  10:30 PM 3/13/2002
#
#  Copyright (c)2002 Actinic Software Ltd
#
############################################################

sub RemoveTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	my $sTagID;
	if( $ParameterHash->{TAG} )					# VALUE contains HREF
		{
		$sTagID = $ParameterHash->{TAG};		# use it
		}
	my $sXMLTag = $ACTINIC::B2B->GetXML($sTagID);
	if( ref($sInsideText) && !$sXMLTag)
		{
		$$sInsideText = "";
		}
	return "";
	}

############################################################
#
#  sub BaseHrefTagHandler
#  Insert BASE HREF tag so that it is only present in dynamic pages
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Zoltan Magyar
#
#   Copyright (c) Actinic Software Ltd (2000)
#
############################################################

sub BaseHrefTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	if( $sTag =~ /^\// )									# If not End tag
		{
		return "";
		}
	my $sReplace;
	my $sURL;
	if (defined $::Session)
		{
		$sURL = $::Session->GetBaseUrl();
		}
	#
	# Separated CGI and catalog servers and the store branding are
	# mutually exclusive. So don't care store branding feature when
	# SSL is used for login and checkout pages only
	#
	if ($$::g_pSetupBlob{'USE_SSL'})							# if SSL is used for any pages
		{
		$sReplace = $$::g_pSetupBlob{SSL_CATALOG_URL};	# use the SSL Catalog base href
		if ($$::g_pSetupBlob{'SSL_USEAGE'} == "1")		# if SSL is used for only some pages
			{
			my $sScriptFileName = ACTINIC::GetScriptFileName();
			if ($::g_bJustAfterLogin)
				{
				if ($::Session->GetDigest() ne '' && $ParameterHash->{VALUE})
					{
					$sReplace = $ParameterHash->{VALUE};								# use it
					}
				}
			elsif ($sScriptFileName !~ /^ma/ &&											# if this is not a my account script
				$::g_InputHash{ACTION} ne ACTINIC::GetPhrase(-1, 2771) &&	# and not forgot password
				$::g_InputHash{ACTION} ne 'PWDCH' &&								# and not change password page
				!$::g_InputHash{CHANGEDPASSWORD} &&									# and not change password page
				$ParameterHash->{VALUE})												# and a value has been supplied
				{
				$sReplace = $ParameterHash->{VALUE};								# use it
				}
			}
		elsif ($ParameterHash->{VALUE})					# if we have a value as a parameter
			{
			$sReplace = $ParameterHash->{VALUE};		# use it
			}
		}
	#
	# Otherwise try to detemine the correct BASE HREF value to
	# maintain the store branding feature.
	#
	else
		{
		if ($sURL)										# if non script page found use it
			{
			$sReplace = $sURL;							# use referer
			}
		elsif( $ParameterHash->{VALUE} )				# VALUE contains HREF
			{
			$sReplace = $ParameterHash->{VALUE};	# use it
			}
		#
		# If it is the Brochure index page uploaded above acatalog
		# the fix up the URL by removing acatalog/
		#
		if ( $ParameterHash->{FORCED} )
			{
			my $StoreFolderName = ACTINIC::GetStoreFolderName();
			$sReplace =~ s/$StoreFolderName\/$//;
			}
		}
	$$sInsideText = '<BASE HREF="' . $sReplace . '">';	# Insert the tag

	return "";
	}

############################################################
#
#  NotInB2BTagHandler
#  If there is a registered user removes the tags end text between
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Ryszard Zybert  Jul 28 14:20:06 BST 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
#
############################################################

sub NotInB2BTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag !~ /^\// )																						# If not End tag
		{
		my $sDigest = $ACTINIC::B2B->Get('UserDigest');
		if( $sDigest )			# If there is a user remove tags and text
			{
			if( ref($sInsideText) )
				{
				$$sInsideText = "";
				}
			}
		}
	return "";
	}

############################################################
#
#  OnlyInB2BTagHandler
#  If there is no registered user then removes the tags end text between
#  Obviously this tag can only be used on dynamic pages
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Zoltan Magyar
#
#  Copyright (c) Actinic Software Ltd (2010)
#
############################################################

sub OnlyInB2BTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag !~ /^\// )																						# If not End tag
		{
		my $sDigest = $ACTINIC::B2B->Get('UserDigest');
		if( !$sDigest )			# If there is a user remove tags and text
			{
			if( ref($sInsideText) )
				{
				$$sInsideText = "";
				}
			}
		}
	return "";
	}
	
############################################################
#
#  UnregTagHandler - callback for UNREG tag
#  If there is a registered user removes the tags end text between
#  Otherwise produces a warning page and bounces to login page
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Ryszard Zybert  May 16 15:23:37 BST 2000
#
#  Copyright (c) Actinic Software Ltd (1999)
#
############################################################

sub UnregTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	my $sDigest = $ACTINIC::B2B->Get('UserDigest');

	if( $sTag =~ /^\// )																						# If not End tag
		{
		return "";
		}
	#
	# When we get here by an invalid user or password from the login page on a closed B2B site
	# the $sDigest won't be defined and we get "unregistered customers not allowed" error
	# instead of "invalid user" message.
	# So lets be tricky and check this case as well.
	#
	if( $sDigest ||										# If there is a user remove tags and text
		 $::g_bLoginPage )								# or invalid user or password
		{
		if( ref($sInsideText) ) { $$sInsideText = ""; }
		return "";
		}
	elsif ($::prog_name ne "SearchScript" && $::prog_name ne "SearchHighligh")
		{
		#
		# If we were here already then the processing should be stopped here
		# to avoid infinitive loop
		#
		if ($::g_RECURSION_ACTIVE)
			{
			return "";
			}
		#
		# This will only have effect if JavaScript is disabled.
		# Otherwise there is a JavaScript alert and this script is no called
		# So - in this case we just show warning and jup back to original page
		#
		my ($Status, $sError, $sHTML) = ACTINIC::ReturnToLastPage(7," " ,
																					 ACTINIC::GetPhrase(-1, 208),
																					$::g_sWebSiteUrl,
																					$::g_sContentUrl, $::g_pSetupBlob, %::g_InputHash);
		if ($Status != $::SUCCESS)						# If even this didn't work - we give up - there is an error
			{
			ACTINIC::ReportError($sError, ACTINIC::GetPath());
			}
		#
		# ACTINIC::PrintPage calls the parser again
		# We should make sure this function won't be called recursively
		# so se a flag here and check above
		#
		$::g_RECURSION_ACTIVE = $::TRUE;
		ACTINIC::PrintPage($sHTML, undef, $::TRUE);	# Print warning page and exit
		exit;
		}
	return "";
	}

############################################################
#
#  PriceTagHandler - price tag callback
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Ryszard Zybert  Dec  7 21:06:24 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
#
############################################################

sub PriceTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if( $sTag =~ /^\// )									# If not End tag
		{
		return "";
		}
	
	if( !$Self->{CurrentSectionBlob} ||				# If section blob not set
		 !$ACTINIC::B2B->Get('UserDigest') )		# or not Business customer
		{
		return "";											# then nothing to do
		}
		
	my @Response;
	#
	# We need the tax information to calculate the prices
	#
	if (!$ActinicOrder::bTaxDataParsed)
		{
		#
		# read the tax blob
		#
		@Response = ACTINIC::ReadTaxSetupFile(ACTINIC::GetPath());
		if ($Response[0] != $::SUCCESS)
			{
			return (@Response);
			}
		ActinicOrder::ParseAdvancedTax();
		}
	@Response = ACTINIC::GetProduct($ParameterHash->{PROD_REF}, $Self->{CurrentSectionBlob},
												ACTINIC::GetPath());	# get this product object
	my ($Status, $Message, $pProduct) = @Response;
	if ($Status != $::SUCCESS)
		{
		return "";											# if any problem, forget it
		}			
	#
	# If we have prices defined for the product, display them
	#
	if (defined $$pProduct{PRICES})
		{
		@Response = ActinicOrder::GetProductPricesHTML($pProduct, undef, $Self->{CurrentSectionBlob});
		$$sInsideText = ($Response[0] != $::SUCCESS) ? $Response[1] : $Response[2];
		}

	return "";																									# Always remove tag
	}

############################################################
#
#  sub SectionTagHandler - section tag callback
#
#  Note: <Actinic:SECTION BLOB="blob file name"/> acts globally
#        $Self->{CurrentSectionBlob} is set here and kept
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Ryszard Zybert  Dec  20 21:06:24 GMT 1999
#
#  Copyright (c) Actinic Software Ltd (1999)
#
############################################################

sub SectionTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId) = @_;

	if( $sTag !~ /^\// )									# If not End tag
		{
		$Self->{CurrentSectionBlob} = $ParameterHash->{BLOB};
		}
	return "";												# Always remove tag
	}

############################################################
#
#  sub MainFrameTagHandler
#  replace SRC parameter in a FRAME tag
#
#  If MAINFRAME XML variable is defined and SRC=name is found
#  in inside text, name will be replaced by the value of MAINFRAME XML
#  variable.
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Ryszard Zybert  Jul 18 11:26:08 BST 2000
#
#  Copyright (c) Actinic Software Ltd (2000)
#
############################################################

sub MainFrameTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId) = @_;

	if( $sTag =~ /^\// )									# If not End tag
		{
		return "";
		}

	my $sXMLTag;
	if( $::g_InputHash{MAINFRAMEURL} )
		{
		$sXMLTag = $::g_InputHash{MAINFRAMEURL};
		}
	else
		{
		$sXMLTag = $ACTINIC::B2B->GetXML("MAINFRAMEURL");
		}
	if( defined($sXMLTag) )								# Replace SRC parameter by tag value
		{
		if( ref($sInsideText) )
			{
			if( $sXMLTag !~ /^((http(s?):)|(\/))/ )
				{
				#
				# Be sure we don't get frames in frames
				#
				if( $sXMLTag eq $$::g_pSetupBlob{FRAMESET_PAGE} )	# for catalog
					{
					$sXMLTag = $$::g_pSetupBlob{CATALOG_PAGE};
					}
				if( $sXMLTag eq $$::g_pSetupBlob{BROCHURE_FRAMESET_PAGE} )	# for brochure
					{
					$sXMLTag = $$::g_pSetupBlob{BROCHURE_MAIN_PAGE};
					}
				#
				# We should use absolute URLs here
				#
				$sXMLTag = $::g_sAccountScript . '?' . ($::g_InputHash{SHOP} ?	'SHOP=' . $::g_InputHash{SHOP} . "&" : "") . 'PRODUCTPAGE=' . $sXMLTag;
				}
			$$sInsideText =~ s/(\s+SRC\s*=\s*)((\"[^\"]+\")|([^\ \>]+))((\s+)|(\>+))/$1\"$sXMLTag\"$5/is;
			}
		}
	return "";												# Always remove tag
	}

############################################################
#
# LocationTagHandler - callback for LOCATION tag
#
# Displays the location in the appropriate form
#
#  Input	 : $sTag - tag name
#           $rsInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
############################################################

sub LocationTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if ( $sTag =~ /^\// )
		{
		return "";
		}

	my $sType = $ParameterHash->{TYPE};
	my $sDigest = $ACTINIC::B2B->Get('UserDigest');
	if ($sDigest eq '')											# if this is a registered buyer
		{
		if ($sType =~ /SELECT/)									# if this is a select tag as opposed to a js tag
			{
			SetLocationSelection($sType, $rsInsideText);	# restore selections
			}
		return '';
		}
	#
	# Get the lists of valid customer account buyer addresses
	#
	my ($plistValidInvoiceAddresses, $plistValidDeliveryAddresses,
		$nInvoiceID, $nDeliveryID);
	$nInvoiceID = -1;
	$nDeliveryID = -1;
	my ($Status, $sMessage, $pBuyer, $pAccount) = ACTINIC::GetBuyerAndAccount($sDigest, ACTINIC::GetPath());
	if ($Status != $::SUCCESS)
		{
		ACTINIC::ReportError($sMessage, ACTINIC::GetPath());
		}
	($Status, $sMessage,
		$plistValidInvoiceAddresses, $plistValidDeliveryAddresses,
		$nInvoiceID, $nDeliveryID) =
			ACTINIC::GetCustomerAddressLists($pBuyer, $pAccount, $::TRUE);
	if ($Status != $::SUCCESS)
		{
		ACTINIC::ReportError($sMessage, ACTINIC::GetPath());
		}
	#
	# Get the appropriate html or js for the type attribute
	#
	# We only adjust state maps if buyer can only select and address
	#
	if ($sType eq 'InvoiceCountryStateMap' &&
		$pBuyer->{'InvoiceAddressRule'} == 1)
		{
		#
		# Populate js state map with customer selections
		#
		$$rsInsideText =
			ActinicOrder::GetBuyerCountryStatesMap($plistValidInvoiceAddresses, 'g_mapInvoiceCountryStateMap');
		}
	elsif ($sType eq 'DeliveryCountryStateMap' &&
		$pBuyer->{'DeliveryAddressRule'} == 1)
		{
		$$rsInsideText =
			ActinicOrder::GetBuyerCountryStatesMap($plistValidDeliveryAddresses, 'g_mapDeliveryCountryStateMap');
		}
	elsif ($sType =~ /^INVOICESELECT/)				# if this an invoice country or state dropdown?
		{
		GetLocationHTML($pBuyer, $sType, $plistValidInvoiceAddresses, $nInvoiceID, $rsInsideText);
		}
	elsif ($sType =~ /^DELIVERSELECT/)				# if this a delivery country or state dropdown?
		{
		GetLocationHTML($pBuyer, $sType, $plistValidDeliveryAddresses, $nDeliveryID, $rsInsideText);
		}
	return('');
	}

############################################################
#
# GetLocationHTML - Restore selections for country or state dropdowns
#
# Input: $pBuyer					- ref to buyer hash
#			$sType					- type from xml tag
#			$plistValidAddresses - ref to list of valid addresses
#			$nDefaultAddressID	- default address ID (for hidden input) or -1 for select address or country/state
#			$rsInsideText			- ref to tag content
#
# Output: $rsInsideText	- html to use
#
# Author:	Mike Purnell
#
############################################################

sub GetLocationHTML
	{
	my ($pBuyer, $sType, $plistValidAddresses, $nDefaultAddressID, $rsInsideText) = @_;
	#
	# Set variables appropriate to xml type attribute
	#
	my ($nAddressRule, $sSelectName, $sLocationKeyPrefix);
	if ($sType =~ /^INVOICE/)
		{
		$nAddressRule = $pBuyer->{InvoiceAddressRule};
		$sLocationKeyPrefix = 'INVOICE';
		$sSelectName = 'LocationInvoice';
		}
	else
		{
		$nAddressRule = $pBuyer->{DeliveryAddressRule};
		$sLocationKeyPrefix = 'DELIVERY';
		$sSelectName = 'LocationDelivery';
		}
	$sSelectName .= ($sType =~ /COUNTRY$/) ? 'Country' : 'Region';
	#
	# Now get html depending upon the addresses
	#
	if ($nDefaultAddressID != -1)						# if we display a hidden input
		{
		my $pSingleAddress = $plistValidAddresses->[0];
		my $sNonEditableFormat = ACTINIC::GetPhrase(-1, 2066);
		if ($sType =~ /COUNTRY$/)						# if country input
			{
			$$rsInsideText = sprintf($sNonEditableFormat,
				ACTINIC::GetCountryName($pSingleAddress->{CountryCode}),
				$sSelectName,
				$pSingleAddress->{CountryCode});
			}
		else													# else state input
			{
			my $sStateName = ACTINIC::GetCountryName($pSingleAddress->{StateCode});
			$$rsInsideText = sprintf($sNonEditableFormat,
				($sStateName ne '') ? $sStateName : '',
				$sSelectName,
				($sStateName ne '') ? $pSingleAddress->{StateCode} : $ActinicOrder::UNDEFINED_REGION);
			}
		return;
		}
	#
	# Get the default address from buyer hash overriding it if they have previously made a selection
	# or they have entered any location information
	#
	if ($sType =~ /^INVOICE/)
		{
		$nDefaultAddressID = GetSelectedBuyerAddress('Invoice', $plistValidAddresses, $pBuyer->{InvoiceAddressID});
		}
	else
		{
		$nDefaultAddressID = GetSelectedBuyerAddress('Delivery', $plistValidAddresses, $pBuyer->{DeliveryAddressID});
		}
	#
	# Get location selection depending upon address rule
	#
	if ($nAddressRule == 1)								# if they can't enter an address, just populate with valid selections
		{
		my ($Status, $sMessage, $sCountryHTML, $sStateHTML) =
			ActinicOrder::GetBuyerLocationOptions($plistValidAddresses,
				$sLocationKeyPrefix, $nDefaultAddressID);
				
		if ($sType =~ /COUNTRY$/)						# if this is country selection?
			{
			$$rsInsideText =~ s/(<select[^>]*>).*(<\/select>)/$1$sCountryHTML$2/is;
			}
		else													# else state selection
			{
			$$rsInsideText =~ s/(<select[^>]*>).*(<\/select>)/$1$sStateHTML$2/is;
			}
		}
	else														# they can enter an address so we just restore the selection
		{
		#
		# If we don't have any location information, use default address locations
		#
		if (!exists $::g_LocationInfo{$sLocationKeyPrefix . '_COUNTRY_CODE'})
			{
			my $pAddress;
			foreach $pAddress (@$plistValidAddresses)
				{
				if ($pAddress->{'ID'} eq $nDefaultAddressID)
					{
					$::g_LocationInfo{$sLocationKeyPrefix . '_COUNTRY_CODE'} = $pAddress->{CountryCode};
					$::g_LocationInfo{$sLocationKeyPrefix . '_REGION_CODE'} = $pAddress->{StateCode};
					last;
					}
				}
			}
		SetLocationSelection($sType, $rsInsideText);
		}
	}

############################################################
#
# SetLocationSelection - Restore selections for country or state dropdowns
#
#  Input: $sType			- type 'INVOICE' or 'DELIVER'
#         $rsInsideText - ref to html
#
#  Returns: full select html
#
#  Author:	Mike Purnell
#
############################################################

sub SetLocationSelection
	{
	my ($sType, $rsInsideText) = @_;
	my $sLocationCode = ($sType =~ /^DELIVER/) ? 'DELIVERY_' : 'INVOICE_';
	$sLocationCode .= ($sType =~ /COUNTRY$/) ? 'COUNTRY_CODE' : 'REGION_CODE';
	my $sSelected = $::g_LocationInfo{$sLocationCode};
	$$rsInsideText =~ s/(<option\s+value\s*=\s*('|")$sSelected('|"))/$1 selected/is;
	if (!$1)
		{
		$sSelected = '---';
		$$rsInsideText =~ s/(<option\s+value\s*=\s*('|")$sSelected('|"))/$1 selected/is;
		}
	}

############################################################
#
# ExtraFooterTagHandler - callback for ExtraFooter tag
#
# Displays the extra footer in the appropriate form
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Mike Purnell
#
############################################################

sub ExtraFooterTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag !~ /^\// )
		{
		if(ref($sInsideText))
			{
			my $nSSPProviderID;
			my $sTemplate = $$sInsideText;
			$$sInsideText = '';
			#
			# For each shipping provider display their trademark text + logo
			#
			my %hTrademarkProviderIDs;
			if ($ACTINIC::B2B->GetXML('ShippingDisclaimingDisplayed') == $::TRUE)											# we are on the shipping charge checkout page
				{
				%hTrademarkProviderIDs = (%::s_Ship_hShippingClassProviderIDs, %::s_Ship_hBasePlusPerProviderIDs); # collect shipping class providers and base-plus-per providers
				}
			if ($::s_Ship_bDisplayExtraCartInformation &&																			# the shipping cost is calculated by a shipping provider
				 $::s_Ship_nSSPProviderID != -1)
				{
				$hTrademarkProviderIDs{$::s_Ship_nSSPProviderID} = $::TRUE;														# add the actual shipping provider
				}
			foreach $nSSPProviderID (keys %hTrademarkProviderIDs)
				{
				my %hVariables;
				$hVariables{$::VARPREFIX . 'POWEREDBYLOGO'} = $$::g_pSSPSetupBlob{$nSSPProviderID}{'POWERED_BY_LOGO'};
				$hVariables{$::VARPREFIX . 'TRADEMARKS'} = $$::g_pSSPSetupBlob{$nSSPProviderID}{'TRADEMARKS'};
				my @Response = ACTINIC::TemplateString($sTemplate, \%hVariables); # make the substitutions
				my ($Status, $Message, $sLine) = @Response;
				if ($Status != $::SUCCESS)
					{
					#
					# Display nothing in case of an error
					#
					$$sInsideText = '';
					return ('');
					}
				$$sInsideText .= $sLine;
				}
			return ('');
			}
		}
	#
	# Nothing will be displayed at this phase
	#
	$$sInsideText = '';
	return('');
	}

############################################################
#
# ExtraCartTagHandler - callback for EXTRACARTTEXT tag
#
# Displays the extra cart text in the appropriate form
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Tibor Vajda
#
############################################################

sub ExtraCartTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag !~ /^\// )
		{
		if(ref($sInsideText))
			{
			if ($::s_Ship_bDisplayExtraCartInformation &&
				 $::s_Ship_nSSPProviderID != -1 &&
				 $::s_Ship_sOpaqueShipData !~ /BasePlusIncrement/)
				{
				#
				# Diplay the rate disclaimer text + logo for the current shipping provider
				#
				my %hVariables;
				$hVariables{$::VARPREFIX . 'POWEREDBYLOGO'} = $$::g_pSSPSetupBlob{$::s_Ship_nSSPProviderID}{'POWERED_BY_LOGO'};
				$hVariables{$::VARPREFIX . 'RATEDISCLAIMER'} = $$::g_pSSPSetupBlob{$::s_Ship_nSSPProviderID}{'RATE_DISCLAIMER'};
				my @Response = ACTINIC::TemplateString($$sInsideText, \%hVariables); # make the substitutions
				my ($Status, $Message, $sLine) = @Response;
				if ($Status == $::SUCCESS)
					{
					$$sInsideText = $sLine;
					return ('');
					}
				}
			}
		}
	#
	# Nothing will be displayed at this phase
	#
	$$sInsideText = '';
	return('');
	}

############################################################
#
# ExtraCartBasePlusPerTagHandler - callback for EXTRACARTBASEPLUSPERTEXT tag
#
# Displays the extra cart base plus per text in the appropriate form
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Tibor Vajda
#
############################################################

sub ExtraCartBasePlusPerTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag !~ /^\// )
		{
		if(ref($sInsideText))
			{
			if ($::s_Ship_bDisplayExtraCartInformation &&
				 $::s_Ship_nSSPProviderID != -1 &&
				 $::s_Ship_sOpaqueShipData =~ /BasePlusIncrement/)
				{
				#
				# Diplay the rate disclaimer text + logo for the current shipping provider
				#
				my %hVariables;
				$hVariables{$::VARPREFIX . 'BASE_PLUS_PER_RATE_DISCLAIMER'} = $$::g_pSSPSetupBlob{$::s_Ship_nSSPProviderID}{'BASE_PLUS_PER_RATE_DISCLAIMER'};
				my @Response = ACTINIC::TemplateString($$sInsideText, \%hVariables); # make the substitutions
				my ($Status, $Message, $sLine) = @Response;
				if ($Status == $::SUCCESS)
					{
					$$sInsideText = $sLine;
					return ('');
					}
				}
			}
		}
	#
	# Nothing will be displayed at this phase
	#
	$$sInsideText = '';
	return('');
	}

############################################################
#
# ExtraShippingTagHandler - callback for EXTRASHIPPINGTEXT tag
#
# Displays the extra shipping bar text in the appropriate form
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Tibor Vajda
#
############################################################

sub ExtraShippingTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag !~ /^\// )
		{
		if(ref($sInsideText))
			{
			#
			# Remember that we displayed provider info cause we have to display tradmark info as well
			#
			$ACTINIC::B2B->SetXML('ShippingDisclaimingDisplayed', $::TRUE);
			#
			# Init variables
			#
			my $nSSPProviderID;
			my $sTemplate = $$sInsideText;
			$$sInsideText = '';
			#
			# For each shipping class provider display their disclaimer text + logo
			#
			foreach $nSSPProviderID (keys %::s_Ship_hShippingClassProviderIDs)
				{
				my %hVariables;
				$hVariables{$::VARPREFIX . 'POWEREDBYLOGO'} = $$::g_pSSPSetupBlob{$nSSPProviderID}{'POWERED_BY_LOGO'};
				$hVariables{$::VARPREFIX . 'RATEDISCLAIMER'} = $$::g_pSSPSetupBlob{$nSSPProviderID}{'RATE_DISCLAIMER'};
				my @Response = ACTINIC::TemplateString($sTemplate, \%hVariables); # make the substitutions
				my ($Status, $Message, $sLine) = @Response;
				if ($Status != $::SUCCESS)
					{
					#
					# Display nothing in case of an error
					#
					$$sInsideText = '';
					return ('');
					}
				$$sInsideText .= $sLine;
				}
			return ('');
			}
		}
	#
	# Nothing will be displayed at this phase
	#
	$$sInsideText = '';
	return('');
	}

############################################################
#
# BasePlusPerInfoTagHandler - callback for EXTRABASEPLUSPERTEXT tag
#
# Displays the extra base plus per warning text in the appropriate form
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Tibor Vajda
#
############################################################

sub BasePlusPerInfoTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag !~ /^\// )
		{
		if(ref($sInsideText))
			{
			#
			# Remember that we displayed provider info cause we have to display tradmark info as well
			#
			$ACTINIC::B2B->SetXML('ShippingDisclaimingDisplayed', $::TRUE);
			#
			# Init variables
			#
			my $nSSPProviderID;
			my $sTemplate = $$sInsideText;
			$$sInsideText = '';
			#
			# For each base plus per provider display their disclaimer text + logo
			#
			foreach $nSSPProviderID (keys %::s_Ship_hBasePlusPerProviderIDs)
				{
				my %hVariables;
				$hVariables{$::VARPREFIX . 'BASE_PLUS_PER_RATE_DISCLAIMER'} = $$::g_pSSPSetupBlob{$nSSPProviderID}{'BASE_PLUS_PER_RATE_DISCLAIMER'};
				my @Response = ACTINIC::TemplateString($sTemplate, \%hVariables); # make the substitutions
				my ($Status, $Message, $sLine) = @Response;
				if ($Status != $::SUCCESS)
					{
					#
					# Display nothing in case of an error
					#
					$$sInsideText = '';
					return ('');
					}
				$$sInsideText .= $sLine;
				}
			return ('');
			}
		}
	#
	# Nothing will be displayed at this phase
	#
	$$sInsideText = '';
	return('');
	}

############################################################
#
# DefaultTaxZoneMessageTagHandler - callback for DEFAULTTAXZONEMESSAGE tag
#
# Displays the default tax zone message in the appropriate form
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
#  Author :	Mike Purnell
#
############################################################

sub DefaultTaxZoneMessageTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if( $sTag !~ /^\// )
		{
		return "";
		}

	if(ref($sInsideText))
		{
		if($ActinicOrder::s_nContext != $ActinicOrder::FROM_CART)
			{
			return('');
			}
		my ($sMessage, $sLocationDescription);
		#
		# Display message in smallest font
		#
		my $sFontOpen = ACTINIC::GetPhrase(-1, 1967);
		my $sFontClose = ACTINIC::GetPhrase(-1, 1970);
		#
		# Customise the message depending upon delivery or invoice
		#
		if($::g_pTaxSetupBlob->{TAX_BY} != $ActinicOrder::eTaxAlways)
			{
			$sLocationDescription = ACTINIC::GetPhrase(-1, 2084);
			if($::g_pTaxSetupBlob->{TAX_BY} == $ActinicOrder::eTaxByInvoice)
				{
				if(defined $::g_LocationInfo{INVOICE_COUNTRY_CODE} &&
					$::g_LocationInfo{INVOICE_COUNTRY_CODE} ne '' &&
					$::g_LocationInfo{INVOICE_COUNTRY_CODE} ne $ActinicOrder::UNDEFINED_REGION &&
					$::g_LocationInfo{INVOICE_COUNTRY_CODE} ne $ActinicOrder::REGION_NOT_SUPPLIED)
					{
					if(defined $::g_LocationInfo{INVOICE_REGION_CODE} &&
						$::g_LocationInfo{INVOICE_REGION_CODE} ne '' &&
						$::g_LocationInfo{INVOICE_REGION_CODE} ne $ActinicOrder::UNDEFINED_REGION)
						{
						$sLocationDescription = ACTINIC::GetCountryName($::g_LocationInfo{INVOICE_REGION_CODE});
						}
					else
						{
						$sLocationDescription = ACTINIC::GetCountryName($::g_LocationInfo{INVOICE_COUNTRY_CODE});
						}
					}
				}
			else
				{
				if(defined $::g_LocationInfo{DELIVERY_COUNTRY_CODE} &&
					$::g_LocationInfo{DELIVERY_COUNTRY_CODE} ne '' &&
					$::g_LocationInfo{DELIVERY_COUNTRY_CODE} ne $ActinicOrder::UNDEFINED_REGION &&
					$::g_LocationInfo{DELIVERY_COUNTRY_CODE} ne $ActinicOrder::REGION_NOT_SUPPLIED)
					{
					if(defined $::g_LocationInfo{DELIVERY_REGION_CODE} &&
						$::g_LocationInfo{DELIVERY_REGION_CODE} ne '' &&
						$::g_LocationInfo{DELIVERY_REGION_CODE} ne $ActinicOrder::UNDEFINED_REGION)
						{
						$sLocationDescription = ACTINIC::GetCountryName($::g_LocationInfo{DELIVERY_REGION_CODE});
						}
					else
						{
						$sLocationDescription = ACTINIC::GetCountryName($::g_LocationInfo{DELIVERY_COUNTRY_CODE});
						}
					}
				}
			#
			# Customise the message depending upon the location info early flag
			#
			my $bRequestInfoEarly = $$::g_pSetupBlob{'TAX_AND_SHIP_EARLY'};
			my $sMessage = $sFontOpen . sprintf(ACTINIC::GetPhrase(-1, 2083), $sLocationDescription);
			$sMessage .= $sFontClose;
			$$sInsideText = $sMessage;
			}
		else
			{
			$$sInsideText = '';
			}
		}

	return('');
	}

############################################################
#
#  ShowForPriceScheduleTagHandler - process the price schedule tags
#
#  Input			:	$sTag - tag name
#						$sInsideText - reference to text between start and end,
#						$ParameterHash - hash of parameters,
#						$sId - current tag prefix,
#						$sFullTag - full text of current tag;
#
#  Output		:	$sInsideText contains replacement for the tag
#
#  Author		:	Tibor Vajda
#
#  Copyright (c) Actinic Software Ltd (2002)
############################################################

sub ShowForPriceScheduleTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	#
	# There is nothing to do with the End tag
	#
	if ($sTag =~ /^\//)									# If End tag
		{
		return ('');
		}
	#
	# Find the schedule Id of the actual user
	#
	my $nScheduleID = $ActinicOrder::RETAILID;	# current user's schedule id (retail by default)
	my ($Status, $sMessage, $pBuyer, $pAccount);	# helpers
	my $sDigest = $ACTINIC::B2B->Get('UserDigest');	# get the user identifying digest
	if ($sDigest)											# if there is logged in user
		{
		my ($Status, $sMessage, $pBuyer) = ACTINIC::GetBuyer($sDigest, ACTINIC::GetPath()); # look up the buyer
		if ($Status == $::SUCCESS)						# we found the buyer info associated to this user
			{
			($Status, $sMessage, $pAccount) = ACTINIC::GetCustomerAccount($pBuyer->{AccountID}, ACTINIC::GetPath()); # find the account information
			if ($Status == $::SUCCESS)
				{
				$nScheduleID = $pAccount->{PriceSchedule}
				}
			}
		#
		# Handling error
		#
		if ($Status != $::SUCCESS)						# schedule ID couldn't be located for some reason
			{
			ACTINIC::ReportError($sMessage, ACTINIC::GetPath());
			}
		}
	#
	# Determine html text to be printed out
	# - if the xml element contains attribute 'HTML', then use its content
	# - otherwise use the node's text value (which is actually $sInsideText)
	#
	if ($ParameterHash->{'HTML'})
		{
		$$sInsideText = $ParameterHash->{'HTML'};
		}
	#
	# Loop through the specified inclusive schedule ids.
	# If one of them identical to the user's schedule is,
	# - then pass the specified html code back in the $sInsideText reference
	# - otherwise pass back empty string in $sInsideText
	#
	my @aIncludedScheduleIds = split(/,/, $ParameterHash->{'Schedules'}); # included schedule ids are stored in the Schedule attribute of the xml element
	my $nIncludedScheduleId;
	foreach $nIncludedScheduleId (@aIncludedScheduleIds)
		{
		if ($nIncludedScheduleId eq $nScheduleID)
			{
			#
			# User's schedule id is defined,
			# thus we don't clear $sInsideText (this will be inserted into the generated html)
			return '';										# Always return ''
			}
		}
	#
	# If current user's price schedule is not defined,
	# then clear $sInsideText -> nothing will be added to the generated html
	#
	$$sInsideText = '';
	return ('');											# Always return ''
	}

#######################################################
#
# GetTemplateFragment - get a named fragment of the XML template
#
# Params:	0 - XML entity structure
#				1 - fragment name
#
# Returns:	0 - HTML
#
# Author:	Zoltan Magyar, 9:04 PM 3/13/2002
#
#######################################################

sub GetTemplateFragment
	{
	my $pXML = shift;
	my $sFragment = shift;
	#
	# Find the order line HTML
	#
	my $pNode = $pXML->FindNode("XMLTEMPLATE", "NAME", $sFragment);
	if (!$pNode)											# the error is critical here
		{
		# Suppressing Message, but still can be used in debug	
		#? ACTINIC::RecordErrors(ACTINIC::GetPhrase(-1, 2201, $sFragment), ACTINIC::GetPath());
		return("");
		}
	return ($pNode->GetOriginal());
	}

#######################################################
#
# 	AddCookieCheck - Adding cookie checking code
#
#  Input			:	$sTag - tag name
#						$sInsideText - reference to text between start and end,
#						$ParameterHash - hash of parameters,
#						$sId - current tag prefix,
#						$sFullTag - full text of current tag;
#
#  Output		:	$sInsideText contains replacement for the tag
#
#  Author		:	Tibor Vajda
#
#  Copyright (c) Actinic Software Ltd (2002)
############################################################

sub AddCookieCheck
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;


	my $sScript = '';										# stores the cookie checking javascript code if needed

	if ($::bCookieCheckRequired)						# cookie checking javascript should be added to the html
		{
		#
		# Determine the url of the warning page generation script
		#
		my $sCgiUrl = ACTINIC::GetScriptUrl($::sShoppingScriptID);	# shopping script is used to display warning
		$sCgiUrl   .= '?ACTION=COOKIEERROR';		# with this parameter
		#
		# Compose the cookie checking javascript
		#
		$sScript = '<SCRIPT LANGUAGE="JavaScript" TYPE="text/javascript">';
		$sScript .= 'if (document.cookie.indexOf("SELLERDECK_COOKIE_CHECK=") == -1)';	# if SELLERDECK_COOKIE_CHECK cookie is not available for the document
		$sScript .= 'document.location.replace("' . $sCgiUrl . '");';			# navigate to the warning page
		$sScript .= '</SCRIPT>';
		}

	$$sInsideText = $sScript;
	return ('');
	}

#######################################################
#
# AddressDetailTagHandler - Substitute address details
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub AddressDetailTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	my $sType = $ParameterHash->{'TYPE'};																# get the format of address details
	$sTag =~ /^(INVOICE|DELIVER|BOTH)ADDRESSDETAILS$/;												# get the prefix
	my $sAddressPrefix = $1;
	if (ref $rsInsideText)																					# if we have a reference
		{
		if ($sAddressPrefix eq 'BOTH')																	# if we want both, substitute invoice and delivery
			{
			$$rsInsideText = SetAddressVariables('INVOICE', $rsInsideText, $sType);
			$$rsInsideText = SetAddressVariables('DELIVER', $rsInsideText, $sType);
			}
		else
			{
			$$rsInsideText = SetAddressVariables($sAddressPrefix, $rsInsideText, $sType);	# substitute details
			}
		}
	return '';
	}

#######################################################
#
# CheckoutURLTagHandler - Substitute checkout URL
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub CheckoutURLTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		my ($Status, $sCheckoutURL);							# get the URL from the custom VAR
		($Status, $sCheckoutURL) = ACTINIC::IsCustomVarDefined('OrderLinkText');
		#
		# If we were called from the order confirmation page, return to confirmation page
		#
		if (exists $::g_InputHash{'FROM'})
			{
			my @arrFields = split(/\?/, $sCheckoutURL);	# get the URL without params
			my $sParams = 'ACTION=RESUME_CHECKOUT';	# new action
			my $sSequence = $::g_InputHash{'FROM'};
			$sSequence =~ s/^CHECKOUT_//;
			$sParams .= "&SEQUENCE=" . $sSequence .
				"&REFPAGE=" . ACTINIC::EncodeText2($::g_InputHash{'FROM'}, $::FALSE);
			if ($$::g_pSetupBlob{'SSL_USEAGE'} == "1")	# split SSL
				{
				my $sOsUrl = sprintf("%sos%6.6d%s",
						$$::g_pSetupBlob{'SSL_CGI_URL'},
						$$::g_pSetupBlob{CGI_ID},
						$$::g_pSetupBlob{CGI_EXT});

				$sCheckoutURL = $arrFields[0] . '?ACTION=SSLBOUNCE&amp;URL=' . ACTINIC::EncodeText2($sOsUrl . '?' . $sParams, $::FALSE);
				}
			else
				{
				$sCheckoutURL = $arrFields[0] . '?' . $sParams;
				}
			}
		$$rsInsideText = $sCheckoutURL;				# set checkout URL
		}
	return '';
	}

#######################################################
#
# ShippingDetailsTagHandler - Substitute shipping NETQUOTEVARS
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub ShippingDetailsTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		if ($$::g_pSetupBlob{'MAKE_SHIPPING_CHARGE'})	# if we're making a shipping charge
			{
			#
			# Restore values from hash
			#
			my $rhashVariableTable = {};
			$rhashVariableTable->{$::VARPREFIX . 'SHIPPINGCAPTION'}	= $::g_ShipInfo{'SHIPPINGCLASS'};
			$rhashVariableTable->{$::VARPREFIX . 'SHIPPING'}			= $::g_ShipInfo{'SHIPPING'};
			$rhashVariableTable->{$::VARPREFIX . 'SHIPUSERDEFINED'}	= $::g_ShipInfo{'USERDEFINED'};
			#
			# Make the substitutions
			#
			my ($nStatus, $sMessage, $sHTML) = ACTINIC::TemplateString($$rsInsideText, $rhashVariableTable); # make the substitutions	
			$$rsInsideText = ($nStatus != $::SUCCESS) ? $sMessage : $sHTML;
			}
		else
			{
			$$rsInsideText = "";
			}
		}
	return '';
	}

#######################################################
#
# TaxDetailsTagHandler - Hide html or set members for the change tax exemption button
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub TaxDetailsTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	if (ref $rsInsideText)
		{
		#
		# Set how many rows we have to display
		#
		$Self->{'TaxConfirmRows'} = ActinicOrder::GetTaxUserInputCount();	# get how many tax user inputs there are
		if ($Self->{'TaxConfirmRows'} == 0)											# no rows to display
			{
			$$rsInsideText = '';															# clear html
			return;
			}
		$Self->{'TaxChangeButtonEnabled'} = 0;										# we haven't displayed a button yet
		}
	return '';
	}

#######################################################
#
# ConfirmTaxExemptionTagHandler - Substitute tax NETQUOTEVARS or clear html
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub ConfirmTaxExemptionTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		my $nTax = $ParameterHash->{'Tax'};							# get which tax we're handling (1 or 2)
		my $sTaxKey = 'TAX_' . $nTax;									# create key into tax hash
		if (!ActinicOrder::IsTaxExemptionAllowed($sTaxKey))	# if user can't exempt this tax
			{
			$$rsInsideText = '';											# don't display anything
			return;
			}
		#
		# Substititute variables
		#
		my $rhashVariableTable = {};
		if ($::g_TaxInfo{'EXEMPT' . $nTax})			# if user claimed exemption, restore values
			{
			$rhashVariableTable->{$::VARPREFIX . 'TAX' . $nTax . 'EXEMPTED'} =
				($::g_TaxInfo{'EXEMPT' . $nTax} == 1) ? "(Exempted)" : '';
			
			$rhashVariableTable->{$::VARPREFIX . 'TAXEXEMPT' . $nTax . 'DATA'}	= $::g_TaxInfo{'EXEMPT' . $nTax . 'DATA'};
			}
		else													# no user exemption, so clear values
			{
			$rhashVariableTable->{$::VARPREFIX . 'TAX' . $nTax . 'EXEMPTED'}		= '';
			$rhashVariableTable->{$::VARPREFIX . 'TAXEXEMPT' . $nTax . 'DATA'}	= 'None';
			}
		$rhashVariableTable->{$::VARPREFIX . 'TAX' . $nTax.'MESSAGE'} = $$::g_pTaxSetupBlob{$sTaxKey}{TAX_MSG};
		#
		# Make the substitutions
		#
		my ($nStatus, $sMessage, $sHTML) = ACTINIC::TemplateString($$rsInsideText, $rhashVariableTable); # make the substitutions	
		$$rsInsideText = ($nStatus != $::SUCCESS) ? $sMessage : $sHTML;
		}
	return '';
	}

#######################################################
#
# TaxUserDefinedTagHandler - Hide or substitute tax user defined value
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub TaxUserDefinedTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		if (ACTINIC::IsPromptHidden(3, 2))			# if tax user defined is hidden
			{
			$$rsInsideText = '';							# blitz all html
			return;
			}
		#
		# Restore the tax user defined button
		#
		my $rhashVariableTable = {};
		$rhashVariableTable->{$::VARPREFIX . 'TAXUSERDEFINED'}	= $::g_TaxInfo{'USERDEFINED'};
		#
		# Make the substitutions
		#
		my ($nStatus, $sMessage, $sHTML) = ACTINIC::TemplateString($$rsInsideText, $rhashVariableTable); # make the substitutions	
		$$rsInsideText = ($nStatus != $::SUCCESS) ? $sMessage : $sHTML;
		}
	return '';
	}
	
#######################################################
#
# TaxChangeCellTagHandler - Set rowspan on first button cell and hide rest
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub TaxChangeCellTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		if ($Self->{'TaxChangeButtonEnabled'} == 0)										# if we haven't displayed a change button
			{
			$$rsInsideText =~ s/ChangeButtonRowSpan/$Self->{'TaxConfirmRows'}/;	# set button cell to span all columns
			$Self->{'TaxChangeButtonEnabled'} = 1;											# store that we displayed change button
			}
		else
			{
			$$rsInsideText = '';																	# displayed a change button cell so hide this one
			}
		}
	return '';
	}

#######################################################
#
# CouponTagHandler - Display coupon entry
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub CouponTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		#
		# Restore the coupon code
		#
		my $rhashVariableTable = {};
		$rhashVariableTable->{$::VARPREFIX . 'COUPONCODE'}	= ACTINIC::EncodeText2($::g_PaymentInfo{'COUPONCODE'});
		#
		# Make the substitutions
		#
		my ($nStatus, $sMessage, $sHTML) = ACTINIC::TemplateString($$rsInsideText, $rhashVariableTable); # make the substitutions	
		$$rsInsideText = ($nStatus != $::SUCCESS) ? $sMessage : $sHTML;
		}
	return '';
	}

#######################################################
#
# GeneralTagHandler - Display general information fields
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub GeneralTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		#
		# If the phase is done, don't display it
		#
		if (ActinicOrder::IsPhaseComplete($::GENERALPHASE) ||
			ActinicOrder::IsPhaseHidden($::GENERALPHASE) )
			{
			$$rsInsideText = '';
			return '';
			}
		#
		# Restore the general information fields
		#
		my $rhashVariableTable = {};
		$rhashVariableTable->{$::VARPREFIX.'GENERALHOWFOUND'} 	= ACTINIC::EncodeText2($::g_GeneralInfo{'HOWFOUND'});
		$rhashVariableTable->{$::VARPREFIX.'GENERALWHYBUY'} 		= ACTINIC::EncodeText2($::g_GeneralInfo{'WHYBUY'});
		$rhashVariableTable->{$::VARPREFIX.'GENERALUSERDEFINED'} = ACTINIC::EncodeText2($::g_GeneralInfo{'USERDEFINED'});
		$rhashVariableTable->{$::VARPREFIX.'GENERALTITLE'} 		= ACTINIC::GetPhrase(-1, 151);
		#
		# Make the substitutions
		#
		my ($nStatus, $sMessage, $sHTML) = ACTINIC::TemplateString($$rsInsideText, $rhashVariableTable); # make the substitutions	
		$$rsInsideText = ($nStatus != $::SUCCESS) ? $sMessage : $sHTML;
		}
	return '';
	}
	
#######################################################
#
# TermsTagHandler - Display terms and conditions
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub TermsTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		#
		# Restore the coupon code
		#
		my $rhashVariableTable = {};
		ACTINIC::SetCheckStatusNQV(\%::g_BillContact, $rhashVariableTable, 'AGREEDTANDC',	'INVOICEAGREETERMSCONDITIONS');
		#
		# Make the substitutions
		#
		my ($nStatus, $sMessage, $sHTML) = ACTINIC::TemplateString($$rsInsideText, $rhashVariableTable); # make the substitutions	
		$$rsInsideText = ($nStatus != $::SUCCESS) ? $sMessage : $sHTML;
		}
	return '';
	}

#######################################################
#
# PaymentTagHandler - Display payment
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub PaymentTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		#
		# if the phase is done, don't display it
		#
		if (ActinicOrder::IsPhaseComplete($::PAYMENTPHASE) ||
			ActinicOrder::IsPhaseHidden($::PAYMENTPHASE) )
			{
			$$rsInsideText = '';
			return '';
			}
		#
		# Restore the payment fields
		#
		my $rhashVariableTable = {};
		my $bDisplayCardFields = $::FALSE;
		my @arrResponse = ActinicOrder::GetPaymentHTML($rhashVariableTable, \$bDisplayCardFields);
		$Self->{'DisplayCardFields'} = $bDisplayCardFields;
		if ($arrResponse[0] != $::SUCCESS)
			{
			$$rsInsideText = $arrResponse[1];
			return ''; 
			}
		#
		# Make the substitutions
		#
		my ($nStatus, $sMessage, $sHTML) = ACTINIC::TemplateString($$rsInsideText, $rhashVariableTable); # make the substitutions	
		$$rsInsideText = ($nStatus != $::SUCCESS) ? $sMessage : $sHTML;
		}
	return '';
	}

#######################################################
#
# PaymentNoPaymentTagHandler - Hide payment fields if no payment or prices aren't displated
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub PaymentNoPaymentTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		my @arrResponse = $::Session->GetCartObject();					# get the cart
		if ($arrResponse[0] != $::SUCCESS)									# if there's an error
			{
			$$rsInsideText = $arrResponse[1];								# display error
			return '';
			}
		my $pCartObject = $arrResponse[2];
		@arrResponse = $pCartObject->SummarizeOrder($::FALSE);		# get the real order total
		if ($arrResponse[0] != $::SUCCESS)									# if we were successful
			{
			$$rsInsideText = $arrResponse[1];
			return '';
			}
		my $nTotal = $arrResponse[6];
		if ($nTotal == 0 || !$$::g_pSetupBlob{'PRICES_DISPLAYED'})	# if there's nothing to pay or prices aren't displayed
			{
			$$rsInsideText = '';													# don't display payment fields
			}
		}
	return '';
	}

#######################################################
#
# DigitalDownloadInfoTagHandler - Display payment
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub DigitalDownloadInfoTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		my @arrResponse = $::Session->GetCartObject();
		if ($arrResponse[0] != $::SUCCESS)					# general error
			{
			$$rsInsideText = $arrResponse[1];
			return '';
			}
		my $pCartObject = $arrResponse[2];
		my $pCartList = $pCartObject->GetCartList();

		@arrResponse = ACTINIC::GetDigitalContent($pCartObject->GetCartList(), $::TRUE);	# get digital content without conditions
		if ($arrResponse[0] == $::FAILURE)
			{
			$$rsInsideText = $arrResponse[1];
			return '';
			}
		my %hDDLinks = %{$arrResponse[2]};
		if (keys %hDDLinks == 0)							# if there is any DD product
			{
			$$rsInsideText = '';
			return '';
			}

		my $rhashVariableTable = {};
		$rhashVariableTable->{$::VARPREFIX.'DDINFOMESSAGE'} = ACTINIC::GetPhrase(-1, 2274, $$::g_pSetupBlob{'DD_EXPIRY_TIME'});
		#
		# Make the substitutions
		#
		my ($nStatus, $sMessage, $sHTML) = ACTINIC::TemplateString($$rsInsideText, $rhashVariableTable); # make the substitutions	
		$$rsInsideText = ($nStatus != $::SUCCESS) ? $sMessage : $sHTML;
		}
	return '';
	}

#######################################################
#
# PaymentCardFieldsTagHandler - Display or hide credit card fields 
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub PaymentCardFieldsTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		#
		# if the phase is done, don't display it
		#
		if (ActinicOrder::IsPhaseComplete($::PAYMENTPHASE) ||
			ActinicOrder::IsPhaseHidden($::PAYMENTPHASE) )
			{
			$$rsInsideText = '';
			return '';
			}
		if (!$Self->{'DisplayCardFields'})			# if we don't display card fields
			{
			$$rsInsideText = '';							# remove them
			return '';
			}
		#
		# Restore the credit card fields except card type
		#
		my $rhashVariableTable = {};
		$rhashVariableTable->{$::VARPREFIX.'PAYMENTCARDNUMBER'} 	= ACTINIC::EncodeText2($::g_PaymentInfo{'CARDNUMBER'});
		$rhashVariableTable->{$::VARPREFIX.'PAYMENTCARDISSUE'} 	= ACTINIC::EncodeText2($::g_PaymentInfo{'CARDISSUE'});
		$rhashVariableTable->{$::VARPREFIX.'PAYMENTCARDVV2'} 		= ACTINIC::EncodeText2($::g_PaymentInfo{'CARDVV2'});
		my ($sec, $min, $hour, $mday, $mon, $nThisYear, $wday, $yday, $isdst) = localtime(time);	# platform independent time
		$nThisYear += 1900;											# make year AD based

		$rhashVariableTable->{$::VARPREFIX.'PAYMENTSTARTMONTHS'} = GetCardMonths($::g_PaymentInfo{'STARTMONTH'});
		$rhashVariableTable->{$::VARPREFIX.'PAYMENTSTARTYEARS'} = GetCardYears($::g_PaymentInfo{'STARTYEAR'}, $nThisYear, -1);
		
		$rhashVariableTable->{$::VARPREFIX.'PAYMENTEXPMONTHS'} = GetCardMonths($::g_PaymentInfo{'EXPMONTH'});
		$rhashVariableTable->{$::VARPREFIX.'PAYMENTEXPYEARS'} = GetCardYears($::g_PaymentInfo{'EXPYEAR'}, $nThisYear, 1);
		#
		# Make the substitutions
		#
		my ($nStatus, $sMessage, $sHTML) = ACTINIC::TemplateString($$rsInsideText, $rhashVariableTable); # make the substitutions	
		$$rsInsideText = ($nStatus != $::SUCCESS) ? $sMessage : $sHTML;
		}
	return '';
	}

#######################################################
#
# PaymentCardTypeTagHandler - Restore selected card type 
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub PaymentCardTypeTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		$$rsInsideText =~ s/<option\s+(value="$::g_PaymentInfo{'CARDTYPE'}")/<option selected $1/is;
		}
	return '';
	}

#######################################################
#
# GetCardMonths - Get credit card months as html options 
#
# Input:	$sSelect	- value to select
#
# Returns:	$sOptions	- list of month options
#
#######################################################

sub GetCardMonths
	{
	my ($sSelect) = @_;
	
	my ($i, $sOptions);
	$sOptions = (($sSelect eq '') ? '<OPTION SELECTED>' : '<OPTION>') . ACTINIC::GetPhrase(5, 9) . "\n";
	for $i (1 .. 12)
		{
		$sOptions .= ($sSelect == $i) ?
			'<OPTION SELECTED>' : '<OPTION>';
		$sOptions .= sprintf('%2.2d', $i) . "\n";
		}
	return $sOptions;
	}

#######################################################
#
# GetCardYears - Get credit card years as html options 
#
# Input:	$sSelect		- value to select
#			$nThisYear	- this year
#			$nIncrement	- the increment (1 or -1)
#
# Returns:	$sOptions	- list of year options
#
#######################################################

sub GetCardYears
	{
	my ($sSelect, $nThisYear, $nIncrement) = @_;
	
	my ($i, $sOptions);
	$sOptions = (($sSelect eq '') ? '<OPTION SELECTED>' : '<OPTION>') . ACTINIC::GetPhrase(5, 10) . "\n";
	for ($i = 0; $i < 11; $i++)
		{
		my $nYear = $nThisYear + ($i * $nIncrement);
		$sOptions .= ($sSelect == $nYear) ?
			'<OPTION SELECTED>' : '<OPTION>';
		$sOptions .= $nYear . "\n";
		}
	return $sOptions;
	}

#######################################################
#
# SetAddressVariables - Substitute address NETQUOTEVARs
#
# Input:	$sAddressPrefix	- tag name
#			$rsInsideText	- reference to text between start and end,
#			$sType - hash of parameters,
#
# Returns:	$sHTML	- substituted text or error message
#
#######################################################

sub SetAddressVariables
	{
	my ($sAddressPrefix, $rsInsideText, $sType) = @_;
	
	my $rhashAddress = ($sAddressPrefix eq 'INVOICE') ?
		\%::g_BillContact :
		\%::g_ShipContact;								# get reference to the contact hash
	#
	# Populate the variable hash with empty values
	#	
	my $rhashVariableTable = {};
	my $sKey;
	foreach $sKey (@ActinicOrder::arrAddressKeys)
		{
		$rhashVariableTable->{$::VARPREFIX . $sAddressPrefix . $sKey} = '';
		}
	if ($rhashAddress->{'RESIDENTIAL'})
		{
		$rhashAddress->{'RESIDENTIAL'} = 'checked';
		}
	#
	# Copy any values from the contact details
	#
	ACTINIC::CopyHash($rhashAddress, $rhashVariableTable, '', $::VARPREFIX . $sAddressPrefix,  $::TRUE);
	#
	# HTML encode each of the variables (XSS protection)
	#
	foreach $sKey (keys %$rhashVariableTable)
		{
		$rhashVariableTable->{$sKey} = ACTINIC::HTMLEncode($rhashVariableTable->{$sKey})
		}
	#
	# If we want a formatted list, add a line break to each non-empty value
	#
	# TODO: add code to add labels
	#
	if ($sType eq 'FORMATTED')
		{
		foreach $sKey (keys %$rhashVariableTable)
			{
			if ($rhashVariableTable->{$sKey})
				{
				$rhashVariableTable->{$sKey} .= '<br />';
				}
			}
		}
	#
	# Make the substitutions
	#
	my ($nStatus, $sMessage, $sHTML) = ACTINIC::TemplateString($$rsInsideText, $rhashVariableTable); # make the substitutions	
	if ($nStatus != $::SUCCESS)
		{
		return $sMessage;
		}
	return $sHTML;
	}
		
#######################################################
#
# DeliverPostalCodeTagHandler - Restore delivery post code for request info early
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub DeliverPostalCodeTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		$$rsInsideText =~ s/NETQUOTEVAR:DELIVERPOSTALCODE/$::g_ShipContact{'POSTALCODE'}/;
		}
	return '';
	}
	
#######################################################
#
# DeliverResidentialTagHandler - Restore delivery residential for request info early
#
# Input:	$sTag				- tag name
#			$rsInsideText	- reference to text between start and end,
#			$ParameterHash - hash of parameters,
#			$sId				- current tag prefix,
#			$sFullTag		- full text of current tag;
#
# Output:	$rsInsideText - contains replacement for the tag
#
# Returns:	empty string
#
#######################################################

sub DeliverResidentialTagHandler
	{
	my $Self = shift;
	my ($sTag,  $rsInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	if (ref $rsInsideText)
		{
		my $sChecked = ($::g_LocationInfo{DELIVERRESIDENTIAL} eq 'on') ? 'checked' : '';
		$$rsInsideText =~ s/NETQUOTEVAR:DELIVERRESIDENTIAL/$sChecked/;
		}
	return '';
	}

############################################################
#
#  SearchFieldTagHandler
#  Handle a search field tag.
#
#  Attributes:
#		actualtag="name"	- replacement tag for Actinic:SearchField
#		selectname="name"	- name attribute of parent <select> tag when <option> tag being processed
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end,
#           $ParameterHash - hash of parameters,
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
############################################################

sub SearchFieldTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;

	#
	# The Actinic:SearchField tag is used to replace static search fields in dynamic web pages e.g. search results.
	# The value supplied to the script in URL parameters are replicated in the SearchFields, which are renamed to
	# the actual tag for the static control e.g. a text box. The static controls are removed by enclosing them in
	# Actinic:StaticSearchField open/closing tags, which are handled by IgnoreTagHandler above to remove the static 
	# field.
	# 
	# If no value is supplied in the URL parameters for a tag then the default selection or check is preserved.
	#
	# An example of the structure created by the layouts is:
	#
	# <Actinic:StaticSearchField>
	#   <input type="radio" name="TB" checked="checked" value="A" id="TB-A" />
   # </Actinic:StaticSearchField>
	# <Actinic:SearchField actualtag="input" type="radio" name="TB" checked="checked" value="A" id="TB-A" />
	#
	# First check if actualtag is defined. If not there we can't do anything
	#
	if (exists $ParameterHash->{'actualtag'})
		{
		#
		# Grab the actual tag to use, convert it to lc and replace the real tag
		#
		my $sActualTag = lc($ParameterHash->{'actualtag'});
		$sFullTag =~ s/$sId$sTag/$sActualTag/;
		#
		# Decide how a selection is to be made for this tag
		#
		my $sSelectValue = ($sActualTag eq "option") ? "selected" : "checked";
		#
		# Work out the "name" to look for in the URL parameters.
		# Checkboxes and text fields have a name attribute which we can use.
		# If not supplied, look for a "selectname" attribute used for "option" tags
		#
		my $sPropertyName = '';
		if (exists $ParameterHash->{'name'})
			{
			$sPropertyName = $ParameterHash->{'name'};
			}
		if (exists $ParameterHash->{'selectname'})
			{
			$sPropertyName = $ParameterHash->{'selectname'};
			$sFullTag =~ s/selectname\=\"[^\"]*\"//;	# remove selectname attribute; not needed in dynamic control
			}
		$sPropertyName =~ s/\ /\+/g;					# we look for property name in URL so tidy up spaces
		if ($sPropertyName)								# did we get a property name?
			{
			#
			# Special processing for "text" fields as we have to get the value of the URL parameter
			#
			if ($sActualTag eq "input" && lc($ParameterHash->{'type'}) eq "text")
				{
				#
				# Extract the text from the parameters and put it in the control
				#
				my $sValue = $::g_InputHash{$sPropertyName};
				$sValue =~ s/\"/\&quot\;/g;			# replace any double quotes with HTML encoded values
				$sFullTag =~ s/actualtag\=\"[^\"]*\"/value\=\"$sValue\"/;	# replace actualtag attribute with value attribute
				}
			#
			# For other controls, we can only check or uncheck them if a "value" attribute is supplied
			# as that value could be in the URL parameters as name=value
			#
			elsif (exists $ParameterHash->{'value'})
				{
				#
				# Get the value, prepare for lookup in URL, and check if this combination was set.
				# Note that we can't use g_InputHash because that only contains one value for each parameter
				# and there may be multiple values for multi-select list boxes.
				#
				my $sValue = $ParameterHash->{'value'};
				my $sEncodedValue = $sValue;
				$sEncodedValue =~ s/\ /\+/g;
				my $sParameter = $sPropertyName . '=' . $sEncodedValue;
				#
				# Special processing for "submit" buttons as we have to remember the submitted value in a hidden field.
				#
				if ($sActualTag eq "input" && lc($ParameterHash->{'type'}) eq "submit")
					{
					my $sHiddenPropertyName = "hf_$sPropertyName";	# hidden property name
					my $sHiddenFieldTag = "<br/><input type=\"hidden\" name=\"hf_$sPropertyName\" value=\"$sValue\">";
					my $sID = $ParameterHash->{'id'};
					#
					# css style to highlight the submitted button link 
					#
					my $sLinkStyle = "\<style type\=\"text\/css\"\> #SubmitLabel$sID {color:red;} \<\/style\>";
					if (exists $::g_InputHash{$sPropertyName})		# URL parameter contains current property name?
						{
						if ($sValue eq $::g_InputHash{$sPropertyName})	# same property value is submitted?
							{
							#
							# include the hidden field to remember the submit button value
							#
							$sFullTag .= $sHiddenFieldTag;
							#
							# add css style to highlight the submit button link
							#
							$sFullTag .= $sLinkStyle;
							}
						}
					elsif (exists $::g_InputHash{$sHiddenPropertyName} && 
							$sValue eq $::g_InputHash{$sHiddenPropertyName}) # remembered submit value?
						{
						#
						# include the hidden field to remember the submit button value
						#
						$sFullTag.= $sHiddenFieldTag;
						#
						# add css style to highlight the submit button link
						#
						$sFullTag .= $sLinkStyle;
						}
					}
				elsif ($::g_OriginalInputData =~ /\Q$sParameter\E(\&|$)/)	# parameter has to end with & (for next parameter) or end of URL
					{
					$sFullTag =~ s/$sSelectValue\=\"[^\"]*\"//;	# Found it - remove any selected attribute
					$sFullTag =~ s/actualtag\=\"[^\"]*\"/$sSelectValue\=\"$sSelectValue\"/;	# and replace actualtag with selected attribute
					}
				#
				# If this radio/checkbox/option wasn't selected explicitly in the URL parameters, check if 
				# another one was specified. If so, make sure that this one is unselected.
				#
				# If another one isn't chosen, then default selection on this one is left as it is.
				# This means that when searching from quick search, all default selections for properties etc are preserved.
				# 
				elsif ($::g_OriginalInputData =~ /(^|&)\Q$sPropertyName\E\=[^\&]*(\&|$)/)
					{
					$sFullTag =~ s/$sSelectValue\=\"[^\"]*\"//;	# remove selection if another one chosen
					}
				}
			}
		#
		# Finally pick up any tags which weren't processed above, deleting the actualtag attribute
		#
		$sFullTag =~ s/actualtag\=\"[^\"]*\"//;
		}
	#
	# Give back the shiny new tag
	#
	return $sFullTag;										
	}
	
############################################################
#
#  SearchFieldEndTagHandler
#  Handle end of a search field tag
#
#  Input	 : $sTag - tag name
#           $sInsideText - reference to text between start and end, (empty for end tag)
#           $ParameterHash - hash of parameters, (parameters for start tag)
#           $sId - current tag prefix,
#           $sFullTag - full text of current tag;
#
#  Output : replacement for the tag
#
############################################################

sub SearchFieldEndTagHandler
	{
	my $Self = shift;
	my ($sTag,  $sInsideText, $ParameterHash, $sId, $sFullTag) = @_;
	
	#
	# When replacing "option" tags there is a distinct end tag:
	#
	# <Actinic:SearchField actualtag="option"
	#   value="160GB" selectname="S&#95;Size0&#95;1">
   # 160GB
   # </Actinic:SearchField>
   #
   # Therefore this has to be rewritten as well. PXML passes the 
   # ParameterHash from the start tag to an explicit End tag handler
   # so we know what replacement tag to use.
   #
	if (defined $ParameterHash->{'actualtag'})
		{
		my $sActualTag = lc($ParameterHash->{'actualtag'});
		$sFullTag =~ s/$sId$sTag/$sActualTag/;
		}		
	return $sFullTag;										
	}

1;

